% collectResults.m collects the results stored in a common folder.
%
% USAGE:
% ======
% [parameters] = collectResults(foldername)
%
% INPUTS:
% =======
% foldername ... name of folder from which results are collected.
%
% Outputs:
% ========
% parameters ... parameter struct.
%
% 2014/06/12 Jan Hasenauer

function parameters = collectResults(foldername)

%% Initialization
load([foldername '/init'],'parameters');

% Initialize
if isfield(parameters,'MS')
if ~isfield(parameters.MS,'par')
parameters.MS.par0 = nan(parameters.number,parameters.MS.n_starts);
parameters.MS.par = nan(parameters.number,parameters.MS.n_starts);
parameters.MS.logPost0 = nan(parameters.MS.n_starts,1);
parameters.MS.logPost = nan(parameters.MS.n_starts,1);
parameters.MS.gradient = nan(parameters.number,parameters.MS.n_starts);
parameters.MS.hessian  = nan(parameters.number,parameters.number,parameters.MS.n_starts);
parameters.MS.n_objfun = nan(parameters.MS.n_starts,1);
parameters.MS.n_iter = nan(parameters.MS.n_starts,1);
parameters.MS.t_cpu = nan(parameters.MS.n_starts,1);
end
end

%% Collection of data
files = dir(fullfile(foldername,'*.csv'));

% Loop: files
for j = 1:length(files)
    % Read file
    v = csvread([foldername '/' files(j).name]);
    
    % Determine index and fieldname
    fn1 = files(j).name(1);
    fn2 = files(j).name((strfind(files(j).name,'__')+2):(length(files(j).name)-4));

    % Assignment
    switch fn1
        case 'M' % -> Multi-start optimization results
            i = str2num(files(j).name(3:(strfind(files(j).name,'__')-1)));
            switch fn2
                case 'logPost', parameters.MS.logPost(i,1) = v;
                case 'logPost0', parameters.MS.logPost0(i,1) = v;
                case 'par', parameters.MS.par(:,i) = v;
                case 'par0', parameters.MS.par0(:,i) = v;
                case 'gradient', parameters.MS.gradient(:,i) = v;            
                case 'hessian', parameters.MS.hessian(:,:,i) = v;
                case 't_cpu', parameters.MS.t_cpu(i,1) = v;
                case 'n_objfun', parameters.MS.n_objfun(i,1) = v;
                case 'n_iter', parameters.MS.n_iter(i,1) = v;
            end
        case 'P' % -> Profile calculation results
            i = str2num(files(j).name(2:(strfind(files(j).name,'__')-1)));
            switch fn2
                case 'par', parameters.P(i).par = v;
                case 'logPost', parameters.P(i).logPost = v;
                case 'R', parameters.P(i).R = v;
            end
   end
end

%% Analyse progress
disp(['progress of MS = ' num2str(100*sum(~isnan(parameters.MS.par(1,:)))/parameters.MS.n_starts) '%']);

%% Sort and save results 
parameters = sortMultiStarts(parameters);
save([foldername '/result'],'parameters');

%% Visualization
if isfield(parameters,'MS')
    plotMultiStarts(parameters);
end

if isfield(parameters,'P')
    plotProfiles(parameters);
end
