% getMultiStarts.m calculates the maximum a posterior estimate of the
%   parameters of a user-supplied posterior function. Therefore, a 
%   multi-start local optimization is used.
%
% Note: This function can exploit up to (n_start + 1) workers when running
% in 'parallel' mode.
%
% USAGE:
% ======
% [...] = getMultiStarts(parameters,objective_function)
% [...] = getMultiStarts(parameters,objective_function,options)
% [parameters,fh] = getMultiStarts(...)
%
% INPUTS:
% =======
% parameters ... parameter struct containing at least:
%   .number ... number of parameter
%   .guess ... initial guess of parameter
%   .min ... lower bound for parameter values       
%   .max ... upper bound for parameter values       
%   .name = {'name1',...} ... names of the parameters       
%   .init_fun ... function to draw starting points for local
%   	optimization. The function has to have the input structure
%           .init_fun(theta_0,theta_min,theta_max)
%       Alternatively, a latin hypercube or a uniform random sampling can
%       be used by setting the respective options
% objective_function ... objective function to be optimized. This function
%       should possess exactly one input, the parameter vector.
% options ... options of algorithm
%   .obj_type ... type of objective function provided
%       = 'log-posterior' (default) ... algorithm assumes that
%               log-posterior or log-likelihood are provided and perfroms 
%               a maximization of the objective function.
%       = 'negative log-posterior' ... algorithm assumes that negative
%               log-posterior or negative log-likelihood are provided and  
%               perfroms a minimization of the objective function.
%   .comp_type ... type of computations
%       = 'sequential' (default) ... classical sequential (in core) method
%       = 'parallel' ... multi-core method exploiting parfor
%   .fmincon ... options for fmincon (the local optimizer)
%   .n_starts ... number of local optimizations (default = 20).
%   .proposal ... method used to propose starting points
%       = 'latin hypercube' (default) ... latin hypercube sampling
%       = 'uniform' ... uniform random sampling
%       = 'user-supplied' ... user supplied function parameters.init_fun
%   .mode ... output of algorithm
%       = 'visual' (default) ... plots are gnerated which show the progress
%       = 'text' ... optimization results for multi-start is printed on screen
%       = 'silent' ... no output during the multi-start local optimization
%   .fh ... handle of figure in which results are printed. If no
%       handle is provided, a new figure is used.
%   .save ... determine whether results are directly saved
%       = 'false' (default) ... results are not saved
%       = 'true' ... results are stored do an extra folder
%   .foldername ... name of the folder in which results are stored.
%       If no folder is provided, a random foldername is generated.
%
% Outputs:
% ========
% parameters ... updated parameter object containing:
%   .MS ... information about multi-start optimization
%       .par(:,i) ... ith MAP
%       .par0(:,i) ... starting point yielding ith MAP
%       .logPost(i) ... log-posterior for ith MAP
%       .logPost0(i) ... log-posterior for starting point yielding ith MAP
%       .gradient(:,i) ... gradient of log-posterior at ith MAP
%       .hessian(:,:,i) ... hessian of log-posterior at ith MAP
%       .n_objfun(i) ... # objective evaluations used to calculate ith MAP
%       .n_iter(i) ... # iterations used to calculate ith MAP
%       .t_cpu(i) ... CPU time for calculation of ith MAP
% fh ... figure handle
%
% 2012/05/31 Jan Hasenauer
% 2012/07/11 Jan Hasenauer
% 2014/06/11 Jan Hasenauer

% function [parameters,fh] = getMultiStarts(parameters,objective_function,options)
function [parameters,fh] = getMultiStarts(varargin)

%% Chweck inputs and assign default values
if nargin >= 2
    parameters = varargin{1};
    objective_function = varargin{2};
else
    error('optimizeMultiStart.m requires at least two inputs.')
end

% Check parameters:
if ~isfield(parameters,'min') || ~isfield(parameters,'max')
    error('Algorithm requires lower and upper bounds');
else
    parameters.min = parameters.min(:);
    parameters.max = parameters.max(:);
end
if length(parameters.min) ~= length(parameters.max)
	error('Dimension of parameters.min and parameters.max does not agree.');
else
    if max(parameters.min >= parameters.max)
        error('There exists at least one i for which parameters.min(i) >= parameters.max(i).');
    end
end
if ~isfield(parameters,'number')
    parameters.number = length(parameters.min);
else
    if parameters.number ~= length(parameters.min)
        error('Dimension mismatch: parameters.number ~= length(parameters.min).');
    end
end
if isfield(parameters,'guess')
    if ~isempty(parameters.guess);
        parameters.guess = parameters.guess(:);
        if length(parameters.guess) ~= length(parameters.max)
            error('Dimension of parameters.guess does not agree with dimesion of parameters.min and .max.');
        end
    end
end
constr.A = [];
constr.b = [];
constr.Aeq = [];
constr.beq = [];
if isfield(parameters,'constraints')
    parameters.constraints = setdefault(parameters.constraints,constr);
else
    parameters.constraints = constr;
end

% Check initial guess
if ~isfield(parameters,'guess')
    parameters.guess = [];
end

% Check and assign options



options.fmincon = optimset('algorithm','interior-point',...
                           'display','off',...
                           'GradObj','on');
options.comp_type = 'sequential'; % 'parallel';
options.obj_type = 'log-posterior'; % 'negative log-posterior'
options.n_starts = 20; % number of multi-start local optimizations
options.proposal = 'latin hypercube'; % 'uniform','user-supplied'
options.mode = 'visual'; % 'text','silent'
options.save = 'false'; % 'true'
options.foldername = strrep(datestr(now,31),' ','__');
options.fh = [];
options.trace = false;
if nargin == 3
    options = setdefault(varargin{3},options);
end

if(options.trace)
    options.fmincon.OutputFcn = @optimizeMultistartOutput;
end

%% Initialization and figure generation
fh = [];
switch options.mode
    case 'visual'
        if isempty(options.fh)
            fh = figure;
        else
            fh = figure(options.fh);
        end
    case 'text'
        fprintf(' \nOptimization:\n=============\n');
    case 'silent' % no output
        % Force fmincon to be silent.
        options.fmincon = optimset(options.fmincon,'display','off');
end


%% Sampling of starting points
switch options.proposal
    case 'latin hypercube'
        % Sampling from latin hypercube
        par0 = [parameters.guess,...
                bsxfun(@plus,parameters.min,bsxfun(@times,parameters.max - parameters.min,...
                              lhsdesign(options.n_starts - size(parameters.guess,2),parameters.number,'smooth','off')'))];
    case 'uniform'
        % Sampling from latin hypercube
        par0 = [parameters.guess,...
                bsxfun(@plus,parameters.min,bsxfun(@times,parameters.max - parameters.min,...
                              rand(parameters.number,options.n_starts - size(parameters.guess,2))))];
    case 'user-supplied'
        % Sampling from user-supplied function
        par0 = [parameters.guess,...
                parameters.init_fun(parameters.guess,parameters.min,parameters.max,options.n_starts - size(parameters.guess,2))];
end
parameters.MS.n_starts = options.n_starts;
parameters.MS.par0 = par0;

%% Preperation of folder
if strcmp(options.save,'true')
    try
       rmdir(options.foldername,'s'); 
    end
    mkdir(options.foldername);
    save([options.foldername '/init'],'parameters');
end

%% Multi-start local optimization -- SEQUENTIAL
if strcmp(options.comp_type,'sequential')
    
% Initialization
parameters.MS.par = nan(parameters.number,options.n_starts);
parameters.MS.logPost0 = nan(options.n_starts,1);
parameters.MS.logPost = nan(options.n_starts,1);
parameters.MS.gradient = nan(parameters.number,options.n_starts);
parameters.MS.hessian  = nan(parameters.number,parameters.number,options.n_starts);
parameters.MS.n_objfun = nan(options.n_starts,1);
parameters.MS.n_iter = nan(options.n_starts,1);
parameters.MS.t_cpu = nan(options.n_starts,1);
parameters.MS.exitflag = nan(options.n_starts,1);
parameters.MS.par_trace = nan(parameters.number,options.fmincon.MaxIter+1,options.n_starts);
parameters.MS.fval_trace = nan(options.fmincon.MaxIter+1,options.n_starts);

% Loop: Mutli-starts
for i = 1:options.n_starts
    try
        % Evaluation of objective function at starting point
        J_0 = obj(parameters.MS.par0(:,i),objective_function,options.obj_type,i);
        
        % Optimization
        if J_0 < inf
            history.x = NaN(parameters.number,options.fmincon.MaxIter+1);
            history.fval = NaN(options.fmincon.MaxIter+1,1);
            % Optimization using fmincon
            t_cpu_fmincon = cputime;
            [theta,J_opt,flag,results_fmincon,~,gradient_opt,hessian_opt] = ...
                fmincon(@(theta) obj(theta,objective_function,options.obj_type,i),...  % negative log-likelihood function
                                    par0(:,i),...    % initial parameter
                                    parameters.constraints.A  ,parameters.constraints.b  ,... % linear inequality constraints
                                    parameters.constraints.Aeq,parameters.constraints.beq,... % linear equality constraints
                                    parameters.min,...     % lower bound
                                    parameters.max,...     % upper bound
                                    [],options.fmincon);   % options
            t_cpu_fmincon = cputime - t_cpu_fmincon;
            n_objfun_fmincon = results_fmincon.funcCount;
            n_iter_fmincon = results_fmincon.iterations;

            % Assignment
            parameters.MS.logPost(i) = -J_opt;
            parameters.MS.logPost0(i) = -J_0;
            parameters.MS.par(:,i) = theta;
            parameters.MS.gradient(:,i) = gradient_opt;            
            if strcmp(options.fmincon.Hessian,'user-supplied') && max(hessian_opt(:)) == 0
                [~,~,hessian_opt] = obj(theta,objective_function,options.obj_type);
            end
            parameters.MS.hessian(:,:,i) = full(hessian_opt);
            parameters.MS.t_cpu(i) = t_cpu_fmincon;
            parameters.MS.n_objfun(i) = n_objfun_fmincon;
            parameters.MS.n_iter(i) = n_iter_fmincon;
            parameters.MS.exitflag(i) = flag;
            parameters.MS.par_trace(:,:,i) = history.x;
            parameters.MS.fval_trace(:,i) = history.fval;
            
            % Save
            if strcmp(options.save,'true')
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__logPost.csv'],-J_opt,'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__logPost0.csv'],-J_0,'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__par.csv'],theta,'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__par0.csv'],par0(:,i),'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__gradient.csv'],gradient_opt,'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__hessian.csv'],full(hessian_opt),'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__t_cpu.csv'],t_cpu_fmincon,'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__n_objfun.csv'],n_objfun_fmincon,'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__n_iter.csv'],n_iter_fmincon,'delimiter',',','precision',12);
            end
            
            % Output
            switch options.mode
                case 'visual', fh = plotMultiStarts(parameters,fh);
                case 'text', disp(['  ' num2str(i,'%d') '/' num2str(options.n_starts,'%d')]);
                case 'silent' % no output
            end
        end        
    catch error_msg
        disp(['Optimization failed because: ' error_msg.message]);
    end
end

% Assignment
parameters = sortMultiStarts(parameters);

end

%% Multi-start local optimization -- PARALLEL
if strcmp(options.comp_type,'parallel')
    
% Initialization
par = nan(parameters.number,options.n_starts);
logPost0 = nan(options.n_starts,1);
logPost = nan(options.n_starts,1);
gradient = nan(parameters.number,options.n_starts);
hessian  = nan(parameters.number,parameters.number,options.n_starts);
n_objfun = nan(options.n_starts,1);
n_iter = nan(options.n_starts,1);
t_cpu = nan(options.n_starts,1);
exitflag = nan(options.n_starts,1);

% Loop: Mutli-starts
parfor i = 1:options.n_starts
    try
        % Evaluation of objective function at starting point
        J_0 = obj(par0(:,i),objective_function,options.obj_type,i);
        
        % Optimization
        if J_0 < inf
            % Optimization using fmincon
            t_cpu_fmincon = cputime;
            [theta,J_opt,flag,results_fmincon,~,gradient_opt,hessian_opt] = ...
                fmincon(@(theta) obj(theta,objective_function,options.obj_type,i),...  % negative log-posterior function
                                    par0(:,i),...    % initial parameter
                                    parameters.constraints.A  ,parameters.constraints.b  ,... % linear inequality constraints
                                    parameters.constraints.Aeq,parameters.constraints.beq,... % linear equality constraints
                                    parameters.min,...     % lower bound
                                    parameters.max,...     % upper bound
                                    [],options.fmincon);   % options
            t_cpu_fmincon = cputime - t_cpu_fmincon;
            n_objfun_fmincon = results_fmincon.funcCount;
            n_iter_fmincon = results_fmincon.iterations;
            

            % AssignmentF
            logPost(i) = -J_opt;
            logPost0(i) = -J_0;
            par(:,i) = theta;
            gradient(:,i) = gradient_opt;            
            if strcmp(options.fmincon.Hessian,'user-supplied') && max(hessian_opt(:)) == 0
                [~,~,hessian_opt] = objective_function(theta,options.obj_type);
            end
            hessian(:,:,i) = full(hessian_opt);
            t_cpu(i) = t_cpu_fmincon;
            n_objfun(i) = n_objfun_fmincon;
            n_iter(i) = n_iter_fmincon;
            exitflag(i) = flag;
            
            % Save
            if strcmp(options.save,'true')
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__logPost.csv'],-J_opt,'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__logPost0.csv'],-J_0,'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__par.csv'],theta,'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__par0.csv'],par0(:,i),'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__gradient.csv'],gradient_opt,'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__hessian.csv'],full(hessian_opt),'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__t_cpu.csv'],t_cpu_fmincon,'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__n_objfun.csv'],n_objfun_fmincon,'delimiter',',','precision',12);
                dlmwrite([options.foldername '/MS' num2str(i,'%d') '__n_iter.csv'],n_iter_fmincon,'delimiter',',','precision',12);
            end
        end        
    catch error_msg
        disp(['Optimization failed because: ' error_msg.message]);
    end
end

% Assignment
parameters.MS.par0 = par0;
parameters.MS.par = par;
parameters.MS.logPost0 = logPost0;
parameters.MS.logPost = logPost;
parameters.MS.gradient = gradient;
parameters.MS.hessian  = hessian;
parameters.MS.n_objfun = n_objfun;
parameters.MS.n_iter = n_iter;
parameters.MS.t_cpu = t_cpu;
parameters.MS.exitflag = exitflag;
parameters = sortMultiStarts(parameters);

% Output
switch options.mode
%     case 'visual', fh = plotMultiStarts(parameters,fh);
%     case 'text', disp(['  ' num2str(i,'%d') '/' num2str(options.n_starts,'%d')]);
%     case 'silent' % no output
end

end

%% Output
switch options.mode
    case {'visual','text'}, disp('-> Multi-start optimization FINISHED.');
    case 'silent' % no output
end

    function stop = optimizeMultistartOutput(x,optimValues,state)
%                 persistent h
        stop = false;
        switch state
            case 'init'
%                                 h = figure;
            case 'interrupt'
            case 'iter'
%                                 figure(h)
                if optimValues.iteration>0
                    history.fval(optimValues.iteration) = optimValues.fval;
                    history.x(:,optimValues.iteration) = x;
                end
%                                 for j = 1:length(x)
%                                     subplot(length(x),1,j)
%                                     plot((1:optimValues.iteration+1)',history.x(j,1:optimValues.iteration+1)','.-')
%                                     xlabel('iteration')
% %                                     ylabel(parameters.name{j})
%                                 end
                
            case 'done'
%                                 close(h)
        end
    end

end


%% Objetive function interface
% This function is used as interface to the user-provided objective
% function. It adapts the sign and supplies the correct number of outputs.
% Furthermore, it catches errors in the user-supplied objective function.
%   theta ... parameter vector
%   fun ... user-supplied objective function
%   type ... type of user-supplied objective function
function varargout = obj(theta,fun,type,start)
persistent nerror;
try
    if(isempty(nerror(start)))
        nerror(start) = 0;
    end
catch
    nerror(start) = 0;
end
try
    switch nargout
        case 1
            J = fun(theta);
            switch type
                case 'log-posterior'          , varargout = {-J};
                case 'negative log-posterior' , varargout = { J};
            end
        case 2
            [J,G] = fun(theta);
            switch type
                case 'log-posterior'          , varargout = {-J,-G};
                case 'negative log-posterior' , varargout = { J, G};
            end
        case 3
            [J,G,H] = fun(theta);
            switch type
                case 'log-posterior'          , varargout = {-J,-G,-H};
                case 'negative log-posterior' , varargout = { J, G, H};
            end
    end
catch error_msg
    if(strcmp(error_msg.message,'failed to integrate ODE'))
        nerror(start) = nerror(start) + 1;
        if(nerror(start)>10)
            error(['Evaluation of likelihood failed because: ' error_msg.message])
        else
            switch nargout
                case 1
                    varargout = {inf};
                case 2
                    varargout = {inf,zeros(length(theta),1)};
                case 3
                    varargout = {inf,zeros(length(theta),1),zeros(length(theta))};
            end
        end
    else
        warning(['Evaluation of likelihood failed because: ' error_msg.message]);
        switch nargout
            case 1
                varargout = {inf};
            case 2
                varargout = {inf,zeros(length(theta),1)};
            case 3
                varargout = {inf,zeros(length(theta),1),zeros(length(theta))};
        end
    end

end

end

