% generateC generates all the required C files which are required for the compilation of the mex simulation file.
%
% USAGE:
% ======
% generateC( modelname, modelstruct)
%
% INPUTS:
% =======
% modelname | specifies the name of the model which will be later used for the naming of the simualation file
% modelstruct | is the struct generated by parseModel

function cw_generateC(filename,struct)

[odewrap_path,~,~]=fileparts(which('cw_compileC.m'));

if(~exist(fullfile(odewrap_path,'models'),'dir'))
    mkdir(fullfile(odewrap_path,'models'));
    mkdir(fullfile(odewrap_path,'models',filename));
else
    if(~exist(fullfile(odewrap_path,'models',filename),'dir'))
        mkdir(fullfile(odewrap_path,'models',filename))
    end
end


%
%----------------------------------------------------------------
%     filename.c
% function specification
%----------------------------------------------------------------
%

nx = length(struct.sym.x);
ny = length(struct.sym.y);


fid = fopen(fullfile(odewrap_path,'models',filename,[filename '.c']),'w');
fprintf(fid,['#include "' filename '.h"\n']);
fprintf(fid,'#include <cvodes/cvodes.h>\n');
fprintf(fid,'#include <cvodes/cvodes_band.h>\n');
fprintf(fid,'#include <cvodes/cvodes_bandpre.h>\n');
fprintf(fid,'#include <cvodes/cvodes_dense.h>\n');
fprintf(fid,'#include <cvodes/cvodes_sparse.h>\n');
fprintf(fid,'#include <cvodes/cvodes_diag.h>\n');
fprintf(fid,'#include <cvodes/cvodes_spbcgs.h>\n');
fprintf(fid,'#include <cvodes/cvodes_spgmr.h>\n');
fprintf(fid,'#include <cvodes/cvodes_sptfqmr.h>\n');
fprintf(fid,'#include <nvector/nvector_serial.h>\n');
fprintf(fid,'#include <sundials/sundials_types.h>\n');
fprintf(fid,'#include <sundials/sundials_math.h>\n');
fprintf(fid,'#include <symbolic_functions.c>\n');
fprintf(fid,'#include <udata.h>\n');
fprintf(fid,'#include <math.h>\n');
fprintf(fid,'#include <mex.h>\n');
fprintf(fid,'\n');
fprintf(fid,'#define pi 3.141592653589793\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf('xdot | ');
fprintf(fid,[' int xdot_' filename '(realtype t, N_Vector x, N_Vector xdot, void *user_data)\n']);
fprintf(fid,'{\n');
fprintf(fid,'  int ix;\n');
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *qpositivex = data->qpositivex;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
fprintf(fid,'  double *xdot_tmp = N_VGetArrayPointer(xdot);\n');
writeCcode(struct, 'xdot', fid);
fprintf(fid,'\n');
fprintf(fid,['  for (ix=0; ix<' num2str(nx) '; ix++) {\n']);
fprintf(fid,'    if(mxIsNaN(xdot_tmp[ix])) xdot_tmp[ix] = 0.0;\n');
fprintf(fid,'    if(qpositivex[ix]>0.5 && x_tmp[ix]<0.0 && xdot_tmp[ix]<0.0) xdot_tmp[ix] = -xdot_tmp[ix];\n');
fprintf(fid,'  }\n');
fprintf(fid,'\n');
fprintf(fid,'  return(0);\n');
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf('xBdot | ');
fprintf(fid,[' int xBdot_' filename '(realtype t, N_Vector x, N_Vector xB, N_Vector xBdot, void *user_data)\n']);
fprintf(fid,'{\n');
fprintf(fid,'  int ixB;\n');
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
fprintf(fid,'  double *xB_tmp = N_VGetArrayPointer(xB);\n');
fprintf(fid,'  double *xBdot_tmp = N_VGetArrayPointer(xBdot);\n');
writeCcode(struct, 'xB', fid);
fprintf(fid,'\n');
fprintf(fid,['  for (ixB=0; ixB<' num2str(nx*ny) '; ixB++) {\n']);
fprintf(fid,'    if(mxIsNaN(xBdot_tmp[ixB])) xBdot_tmp[ixB] = 0.0;\n');
fprintf(fid,'  }\n');
fprintf(fid,'\n');
fprintf(fid,'  return(0);\n');
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf('xQB | ');
fprintf(fid,[' int xQB_' filename '(realtype t, N_Vector x, N_Vector xB, N_Vector qBdot, void *user_data)\n']);
fprintf(fid,'{\n');
fprintf(fid,'  int iyp;\n');
fprintf(fid,'  int ip;\n');
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  int *plist = data->plist;\n');
fprintf(fid,'  int np = *data->np;\n');
fprintf(fid,'  int ny = *data->ny;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
fprintf(fid,'  double *xB_tmp = N_VGetArrayPointer(xB);\n');
fprintf(fid,'  double *qBdot_tmp = N_VGetArrayPointer(qBdot);\n');
fprintf(fid,['  memset(qBdot_tmp,0,sizeof(double)*' num2str(ny) '*np);\n']);
fprintf(fid,'  for(ip=0; ip<np; ip++) {\n');
fprintf(fid,'  switch (plist[ip]) {\n');
writeCcode_sensi(struct, 'xQB', fid);
fprintf(fid,'  }\n');
fprintf(fid,'  }\n');
fprintf(fid,'\n');
fprintf(fid,['  for (iyp=0; iyp<' num2str(ny) '*np; iyp++) {\n']);
fprintf(fid,'    if(mxIsNaN(qBdot_tmp[iyp])) qBdot_tmp[iyp] = 0.0;\n');
fprintf(fid,'  }\n');
fprintf(fid,'\n');
fprintf(fid,'  return(0);\n');
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf('x0 | ');
fprintf(fid,[' void x0_' filename '(N_Vector x0, void *user_data)\n']);
fprintf(fid,'{\n');
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x0_tmp = N_VGetArrayPointer(x0);\n');
writeCcode(struct, 'x0', fid);
fprintf(fid,'  \n');
fprintf(fid,'  \n');
fprintf(fid,'  return;\n');
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf('Jv | ');
fprintf(fid,[' int Jv_' filename '(N_Vector v, N_Vector Jv, realtype t,\n']);
fprintf(fid,'  	N_Vector x, N_Vector xdot, void *user_data, N_Vector tmp)\n');
fprintf(fid,'{\n');
fprintf(fid,'  int ix;\n');
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
fprintf(fid,'  double *v_tmp = N_VGetArrayPointer(v);\n');
fprintf(fid,'  double *Jv_tmp = N_VGetArrayPointer(Jv);\n');
writeCcode(struct, 'Jv', fid);
fprintf(fid,'\n');
fprintf(fid,['  for (ix=0; ix<' num2str(nx) '; ix++) {\n']);
fprintf(fid,'    if(mxIsNaN(Jv_tmp[ix])) Jv_tmp[ix] = 0.0;\n');
fprintf(fid,'  }\n');
fprintf(fid,'\n');
fprintf(fid,'  return(0);\n');
fprintf(fid,'}\n');
fprintf('JvB | ');
fprintf(fid,[' int JvB_' filename '(N_Vector vB, N_Vector JvB, realtype t,\n']);
fprintf(fid,'  	N_Vector x, N_Vector xB, N_Vector xBdot, void *user_data, N_Vector tmpB)\n');
fprintf(fid,'{\n');
fprintf(fid,'  int ix;\n');
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
fprintf(fid,'  double *xB_tmp = N_VGetArrayPointer(xB);\n');
fprintf(fid,'  double *vB_tmp = N_VGetArrayPointer(vB);\n');
fprintf(fid,'  double *JvB_tmp = N_VGetArrayPointer(JvB);\n');
writeCcode(struct, 'JvB', fid);
fprintf(fid,'\n');
fprintf(fid,['  for (ix=0; ix<' num2str(nx) '; ix++) {\n']);
fprintf(fid,'    if(mxIsNaN(JvB_tmp[ix])) JvB_tmp[ix] = 0.0;\n');
fprintf(fid,'  }\n');
fprintf(fid,'\n');
fprintf(fid,'  return(0);\n');
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf(fid,[' int JBand_' filename '(long int N, long int mupper, long int mlower,']);
fprintf(fid,'   realtype t, N_Vector x, N_Vector xdot,\n');
fprintf(fid,'  	DlsMat J, void *user_data, \n');
fprintf(fid,'  	N_Vector tmp1, N_Vector tmp2, N_Vector tmp3)\n');
fprintf(fid,'{\n');
fprintf(fid,['  return(J_' filename '(N,t,x,xdot,J,user_data,tmp1,tmp2,tmp3));\n']);
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf('J | ');
fprintf(fid,[' int J_' filename '(long int N, realtype t, N_Vector x,\n']);
fprintf(fid,'  	N_Vector xdot, DlsMat J, void *user_data, \n');
fprintf(fid,'  	N_Vector tmp1, N_Vector tmp2, N_Vector tmp3)\n');
fprintf(fid,'{\n');
fprintf(fid,'  int iJ;\n');
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
writeCcode(struct, 'J', fid);
fprintf(fid,'\n');
fprintf(fid,['  for (iJ=0; iJ<' num2str(nx^2) '; iJ++) {\n']);
fprintf(fid,'    if(mxIsNaN(J->data[iJ])) J->data[iJ] = 0.0;\n');
fprintf(fid,'  }\n');
fprintf(fid,'\n');
fprintf(fid,'  return(0);\n');
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf('JSparse | ');
fprintf(fid,[' int JSparse_' filename '(realtype t, N_Vector x,\n']);
fprintf(fid,'  	N_Vector xdot, SlsMat J, void *user_data, \n');
fprintf(fid,'  	N_Vector tmp1, N_Vector tmp2, N_Vector tmp3)\n');
fprintf(fid,'{\n');
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
fprintf(fid,'  SlsSetToZero(J);\n');
for i = 1:length(struct.rowvals)
    fprintf(fid,['  J->rowvals[' num2str(i-1) '] = ' num2str(struct.rowvals(i)) ';\n']);
end
for i = 1:length(struct.colptrs)
    fprintf(fid,['  J->colptrs[' num2str(i-1) '] = ' num2str(struct.colptrs(i)) ';\n']);
end
writeCcode(struct, 'JSparse', fid);
fprintf(fid,'  return(0);\n');
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf(fid,[' int JBBand_' filename '(long int NeqB, long int mupper, long int mlower,']);
fprintf(fid,'   realtype t, N_Vector x, N_Vector xB,\n');
fprintf(fid,'  	N_Vector xdotB, DlsMat J, void *user_data, \n');
fprintf(fid,'  	N_Vector tmp1, N_Vector tmp2, N_Vector tmp3)\n');
fprintf(fid,'{\n');
fprintf(fid,['  return(JB_' filename '(NeqB,t,x,xB,xdotB,J,user_data,tmp1,tmp2,tmp3));\n']);
fprintf(fid,'}\n');
fprintf('JB | ');
fprintf(fid,[' int JB_' filename '(long int N, realtype t, N_Vector x,\n']);
fprintf(fid,'  	N_Vector xB, N_Vector xdotB, DlsMat JB, void *user_data, \n');
fprintf(fid,'  	N_Vector tmp1B, N_Vector tmp2B, N_Vector tmp3B)\n');
fprintf(fid,'{\n');
fprintf(fid,'  int iJ;\n');
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
fprintf(fid,'  double *xB_tmp = N_VGetArrayPointer(xB);\n');
writeCcode(struct, 'JB', fid);
fprintf(fid,'\n');
fprintf(fid,['  for (iJ=0; iJ<' num2str(nx^2) '; iJ++) {\n']);
fprintf(fid,'    if(mxIsNaN(JB->data[iJ])) JB->data[iJ] = 0.0;\n');
fprintf(fid,'  }\n');
fprintf(fid,'\n');
fprintf(fid,'  return(0);\n');
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf('JSparseB | ');
fprintf(fid,[' int JSparseB_' filename '(realtype t, N_Vector x,\n']);
fprintf(fid,'  	N_Vector xB, N_Vector xdotB, SlsMat JB, void *user_data, \n');
fprintf(fid,'  	N_Vector tmp1B, N_Vector tmp2B, N_Vector tmp3B)\n');
fprintf(fid,'{\n');
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
fprintf(fid,'  SlsSetToZero(JB);\n');
for i = 1:length(struct.rowvalsB)
    fprintf(fid,['  JB->rowvals[' num2str(i-1) '] = ' num2str(struct.rowvalsB(i)) ';\n']);
end
for i = 1:length(struct.colptrsB)
    fprintf(fid,['  JB->colptrs[' num2str(i-1) '] = ' num2str(struct.colptrsB(i)) ';\n']);
end
writeCcode(struct, 'JBSparse', fid);
fprintf(fid,'  return(0);\n');
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf('sx | ');
fprintf(fid,[' int sx_' filename '(int Ns, realtype t, N_Vector x, N_Vector xdot,\n']);
fprintf(fid,'  	int ip, N_Vector sx, N_Vector sxdot, void *user_data, \n');
fprintf(fid,'  	N_Vector tmp1, N_Vector tmp2)\n');
fprintf(fid,'{\n');
fprintf(fid,'  int ix;\n');
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  int *plist = data->plist;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
fprintf(fid,'  double *sx_tmp = N_VGetArrayPointer(sx);\n');
fprintf(fid,'  double *sxdot_tmp = N_VGetArrayPointer(sxdot);\n');
fprintf(fid,['  memset(sxdot_tmp,0,sizeof(double)*' num2str(nx) ');\n']);
fprintf(fid,'  switch (plist[ip]) {\n');
writeCcode_sensi(struct,'sx', fid);
fprintf(fid,'  }\n');
fprintf(fid,[' for (ix=0; ix<' num2str(nx) '; ix++) {\n']);
fprintf(fid,'    if(mxIsNaN(sxdot_tmp[ix])) sxdot_tmp[ix] = 0.0;\n');
fprintf(fid,'  }\n');
fprintf(fid,'\n');
fprintf(fid,'  return(0);\n');
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf('sx0 | ');
fprintf(fid,[' void sx0_' filename '(int ip, N_Vector sx0, void *user_data)\n']);
fprintf(fid,'{\n');
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  int *plist = data->plist;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *sx0_tmp = N_VGetArrayPointer(sx0);\n');
fprintf(fid,['  memset(sx0_tmp,0,sizeof(double)*' num2str(nx) ');\n']);
fprintf(fid,'  switch (ip) {\n');
writeCcode_sensi(struct, 'sx0', fid);
fprintf(fid,'  }\n');
fprintf(fid,'\n');
fprintf(fid,'  return;\n');
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf(fid,['void y_' filename '(double t, int nt, int it, double *y, double *p, double *k, double *u, double *x){\n']);
writeCcode(struct, 'y', fid);
fprintf(fid,'    \n');
fprintf(fid,'    return;\n');
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf('dydp | ');
fprintf(fid,['void dydp_' filename '(double t, int nt, int it, double *dydp, double *y, double *p, double *k, double *u, double *x, int *plist, int np, int ny){\n']);
fprintf(fid,'  \n');
fprintf(fid,'  int ip;\n');
fprintf(fid,'  for(ip=0; ip<np; ip++) {\n');
fprintf(fid,['  memset(dydp,0,sizeof(double)*' num2str(ny) '*np*nt);\n']);
fprintf(fid,'  switch (plist[ip]) {\n');
writeCcode_sensi(struct, 'dydp', fid);
fprintf(fid,'  }\n');
fprintf(fid,'  }\n');
fprintf(fid,'  \n');
fprintf(fid,'  return;\n');
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf('dydx | ');
fprintf(fid,['void dydx_' filename '(double t,double *dydx, double *y, double *p, double *k, double *x){\n']);
writeCcode(struct, 'dydx', fid);
fprintf(fid,'  \n');
fprintf(fid,'  return;\n');
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf('sy | ');
fprintf(fid,['void sy_' filename '(double t, int nt, int it, int ip, int np, int nx, int ny, double *sy, double *p, double *k, double *x, double *sx){\n']);
fprintf(fid,'  switch (ip) {\n');
writeCcode_sensi(struct, 'sy', fid);
fprintf(fid,'  }\n');
fprintf(fid,'  \n');
fprintf(fid,'  return;\n');
fprintf(fid,'}\n');
fprintf('root | ');
fprintf(fid,['int root_' filename '(double t, N_Vector x, realtype *gout, void *user_data){\n']);
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
writeCcode(struct, 'roots', fid);
fprintf(fid,'  return(0);\n');
fprintf(fid,'}\n');
fprintf('sroot | ');
fprintf(fid,['double sroot_' filename '(double t, int ip, int ir, N_Vector x, N_Vector sx, void *user_data){\n']);
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
fprintf(fid,'  double *sx_tmp = N_VGetArrayPointer(sx);\n');
fprintf(fid,'  double dr_dp;\n');
fprintf(fid,'  switch (ip) {\n');
writeCcode_sensi(struct, 'sroot', fid);
fprintf(fid,'  }\n');
fprintf(fid,'  return(dr_dp);\n');
fprintf(fid,'}\n');
fprintf('s2root | ');
fprintf(fid,['double s2root_' filename '(double t, int ip, int jp, int ir, N_Vector x, N_Vector sx, void *user_data){\n']);
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
fprintf(fid,'  double *sx_tmp = N_VGetArrayPointer(sx);\n');
fprintf(fid,'  double ddr_dpdp;\n');
fprintf(fid,'  switch (ip) {\n');
writeCcode_sensi(struct, 's2root', fid);
fprintf(fid,'  }\n');
fprintf(fid,'  return(ddr_dpdp);\n');
fprintf(fid,'}\n');
fprintf('srootval | ');
fprintf(fid,['double srootval_' filename '(double t, int ip, int ir, N_Vector x, N_Vector sx, void *user_data){\n']);
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
fprintf(fid,'  double *sx_tmp = N_VGetArrayPointer(sx);\n');
fprintf(fid,'  double dg_dp;\n');
fprintf(fid,'  switch (ip) {\n');
writeCcode_sensi(struct, 'srootval', fid);
fprintf(fid,'  }\n');
fprintf(fid,'  return(dg_dp);\n');
fprintf(fid,'}\n');
fprintf('s2rootval | ');
fprintf(fid,['double s2rootval_' filename '(double t, int ip, int jp, int ir, N_Vector x, N_Vector sx, void *user_data){\n']);
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
fprintf(fid,'  double *sx_tmp = N_VGetArrayPointer(sx);\n');
fprintf(fid,'  double ddg_dpdp;\n');
fprintf(fid,'  switch (ip) {\n');
writeCcode_sensi(struct, 's2rootval', fid);
fprintf(fid,'  }\n');
fprintf(fid,'  return(ddg_dpdp);\n');
fprintf(fid,'}\n');
fprintf('deltadisc | ');
fprintf(fid,['void deltadisc_' filename '(double t, int idisc, N_Vector x, void *user_data){\n']);
fprintf(fid,'  int ix;\n');
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
fprintf(fid,['  double deltadisc[' num2str(nx) '];\n']);
fprintf(fid,['  memset(deltadisc,0,sizeof(double)*' num2str(nx) ');\n']);
writeCcode(struct, 'deltadisc', fid);
fprintf(fid,['  for(ix = 0; ix<' num2str(nx) ';ix++){;\n']);
fprintf(fid,'  x_tmp[ix] += deltadisc[ix];\n');
fprintf(fid,'  };\n');
fprintf(fid,'}\n');
fprintf(fid,['void sdeltadisc_' filename '(double t, int idisc, N_Vector x, N_Vector *sx, void *user_data){\n']);
fprintf(fid,'  int ix;\n');
fprintf(fid,'  int ip;\n');
fprintf(fid,'  UserData data = (UserData) user_data;\n');
fprintf(fid,'  int *plist = data->plist;\n');
fprintf(fid,'  double *p = data->p;\n');
fprintf(fid,'  double *k = data->k;\n');
fprintf(fid,'  double *u = data->u;\n');
fprintf(fid,'  double *x_tmp = N_VGetArrayPointer(x);\n');
fprintf(fid,'  double *sx_tmp;\n');
fprintf(fid,'  int np = *data->np;\n');
fprintf(fid,['  double deltadisc[' num2str(nx) '];\n']);
fprintf(fid,['  double *sdeltadisc;\n']);
fprintf(fid,['  memset(deltadisc,0,sizeof(double)*' num2str(nx) ');\n']);
fprintf(fid,['  sdeltadisc = mxMalloc(sizeof(double)*' num2str(nx) '*np);\n']);
fprintf(fid,['  memset(sdeltadisc,0,sizeof(double)*' num2str(nx) '*np);\n']);
writeCcode(struct, 'deltadisc', fid);
fprintf(fid,'  for (ip=0; ip<np; ip++) {\n');
fprintf(fid,'  sx_tmp = N_VGetArrayPointer(sx[plist[ip]]);\n');
fprintf(fid,'     switch (plist[ip]) {\n');
writeCcode_sensi(struct, 'sdeltadisc', fid);
fprintf(fid,'     }\n');
fprintf(fid,'  }\n');
fprintf(fid,['  for(ip = 0; ip<np;ip++){\n']);
fprintf(fid,'      sx_tmp = N_VGetArrayPointer(sx[plist[ip]]);\n');
fprintf(fid,['      for(ix = 0; ix<' num2str(nx) ';ix++){\n']);
fprintf(fid,'      sx_tmp[ix] += sdeltadisc[plist[ip]+np*ix];\n');
fprintf(fid,'     }\n');
fprintf(fid,'  }\n');
fprintf(fid,['  for(ix = 0; ix<' num2str(nx) ';ix++){\n']);
fprintf(fid,'  x_tmp[ix] += deltadisc[ix];\n');
fprintf(fid,'  };\n');
fprintf(fid,' mxFree(sdeltadisc);\n');
fprintf(fid,'}\n');
fprintf(fid,'\n');
fprintf(fid,'\n');
fprintf('dydp | ');
fprintf(fid,['void dxdotdp_' filename '(double t, int nt, int it, double *dxdotdp, double *p, double *k, double *u, double *x, int *plist, int np, int nx){\n']);
fprintf(fid,'  \n');
fprintf(fid,'  int ip;\n');
fprintf(fid,'  for(ip=0; ip<np; ip++) {\n');
fprintf(fid,'  switch (plist[ip]) {\n');
writeCcode_sensi(struct, 'dxdotdp', fid);
fprintf(fid,'  }\n');
fprintf(fid,'  }\n');
fprintf(fid,'  \n');
fprintf(fid,'  return;\n');
fprintf(fid,'}\n');
fclose(fid);

%
%----------------------------------------------------------------
%     filename.h
% function definitions
%----------------------------------------------------------------
%

fprintf('headers | ');
str_hfile = [...
    '#ifndef _MY_' filename '\n'...
    '#define _MY_' filename '\n'...
    '\n'...
    '#include <cvodes/cvodes.h>\n'...
    '#include <cvodes/cvodes_band.h>\n'...
    '#include <cvodes/cvodes_bandpre.h>\n'...
    '#include <cvodes/cvodes_dense.h>\n'...
    '#include <cvodes/cvodes_sparse.h>\n'...
    '#include <cvodes/cvodes_diag.h>\n'...
    '#include <cvodes/cvodes_spbcgs.h>\n'...
    '#include <cvodes/cvodes_spgmr.h>\n'...
    '#include <cvodes/cvodes_sptfqmr.h>\n'...
    '#include <nvector/nvector_serial.h>\n'...
    '#include <sundials/sundials_types.h>\n'...
    '#include <sundials/sundials_math.h>\n'...
    '#include <udata.h>\n'...
    '#include <math.h>\n'...
    '#include <mex.h>\n'...
    '\n'...
    '             void x0_' filename '(N_Vector x0, void *user_data);\n'...
    '             int J_' filename '(long int N, realtype t, N_Vector x,N_Vector fx, DlsMat J, void *user_data,N_Vector tmp1, N_Vector tmp2, N_Vector tmp3);\n'...
    '             int JSparse_' filename '(realtype t, N_Vector x,N_Vector fx, SlsMat J, void *user_data,N_Vector tmp1, N_Vector tmp2, N_Vector tmp3);\n'...
    '             int JBand_' filename '(long int N, long int mupper, long int mlower, realtype t, N_Vector x,N_Vector fx, DlsMat J, void *user_data,N_Vector tmp1, N_Vector tmp2, N_Vector tmp3);\n'...
    '             int JB_' filename '(long int NeqB,realtype t, N_Vector x, N_Vector xB, N_Vector fx, DlsMat J, void *user_data,N_Vector tmp1, N_Vector tmp2, N_Vector tmp3);\n'...
    '             int JSparseB_' filename '(realtype t, N_Vector x, N_Vector xB, N_Vector fx, SlsMat J, void *user_data,N_Vector tmp1, N_Vector tmp2, N_Vector tmp3);\n'...
    '             int JBBand_' filename '(long int N, long int mupper, long int mlower, realtype t, N_Vector x, N_Vector xB, N_Vector fx, DlsMat J, void *user_data,N_Vector tmp1, N_Vector tmp2, N_Vector tmp3);\n'...
    '             int Jv_' filename '(N_Vector v, N_Vector Jv, realtype t, N_Vector x, N_Vector xdot, void *user_data, N_Vector tmp);\n'...
    '             int JBv_' filename '(N_Vector v, N_Vector Jv, realtype t, N_Vector x, N_Vector xdot, void *user_data, N_Vector tmp);\n'...
    '             int sx_' filename '(int Ns, realtype t, N_Vector x, N_Vector xdot,int ip, N_Vector sx, N_Vector sxdot, void *user_data, N_Vector tmp1, N_Vector tmp2);\n'...
    '             void sx0_' filename '(int ip, N_Vector sx0, void *user_data);\n'...
    '             void y_' filename '(double t, int nt, int it, double *y, double *p, double *k, double *u, double *x);\n'...
    '             void dydp_' filename '(double t, int nt, int it, double *dydp, double *y, double *p, double *k, double *u, double *x, int *plist, int np, int ny);\n'...
    '             void dydx_' filename '(double t, double *dydx, double *y, double *p, double *k, double *x);\n'...
    '             void sy_' filename '(double t, int nt, int it, int ip, int np, int nx, int ny, double *sy, double *p, double *k, double *x, double *sx);\n'...
    '             double sroot_' filename '(double t, int ip, int ir, N_Vector x, N_Vector sx, void *user_data);\n'...
    '             double srootval_' filename '(double t, int ip, int ir, N_Vector x, N_Vector sx, void *user_data);\n'...
    '             double s2root_' filename '(double t, int ip, int jp, int ir, N_Vector x, N_Vector sx, void *user_data);\n'...
    '             double s2rootval_' filename '(double t, int ip, int jp, int ir, N_Vector x, N_Vector sx, void *user_data);\n'...
    '             void deltadisc_' filename '(double t, int idisc, N_Vector x, void *user_data);\n'...
    '             void sdeltadisc_' filename '(double t, int idisc, N_Vector x, N_Vector *sx, void *user_data);\n'...
    '\n'...
    '\n'...
    '#endif /* _MY_' filename ' */\n'...
    ];

% write the string to the file
fid = fopen(fullfile(odewrap_path,'models',filename,[filename '.h']),'w');
fprintf(fid,str_hfile);
fclose(fid);

%
%----------------------------------------------------------------
%     cvodewrapfunctions.c
% this file does the dynamic mapping of the universal function calls from
% cvodewrap.c to individual model function calls
%----------------------------------------------------------------
%

fprintf('functionfile | ');
str_funcfile = [...
    '#include "' filename '.c"\n'...
    '                \n'...
    '                int cvodewrap_init(void *cvode_mem, N_Vector x, double t){\n'...
    '                    return CVodeInit(cvode_mem, xdot_' filename ', RCONST(t), x);\n'...
    '                }\n'...
    '                int cvodewrap_binit(void *cvode_mem, int which, N_Vector xB, double t){\n'...
    '                    return CVodeInitB(cvode_mem, which, xBdot_' filename ', RCONST(t), xB);\n'...
    '                }\n'...
    '                int cvodewrap_qbinit(void *cvode_mem, int which, N_Vector xQB){\n'...
    '                    return CVodeQuadInitB(cvode_mem, which, xQB_' filename ', xQB);\n'...
    '                }\n'...
    '                \n'...
    '                void fx0(N_Vector x0, void *user_data){\n'...
    '                    UserData data = (UserData) user_data;\n'...
    '                    x0_' filename '(x0, data);\n'...
    '                }\n'...
    '                \n'...
    '                int cvodewrap_SetDenseJacFn(void *cvode_mem){\n'...
    '                    return CVDlsSetDenseJacFn(cvode_mem, J_' filename ');\n'...
    '                }\n'...
    '                int cvodewrap_SetSparseJacFn(void *cvode_mem){\n'...
    '                    return CVSlsSetSparseJacFn(cvode_mem, JSparse_' filename ');\n'...
    '                }\n'...
    '                int cvodewrap_SetBandJacFn(void *cvode_mem){\n'...
    '                    return CVDlsSetBandJacFn(cvode_mem, JBand_' filename ');\n'...
    '                }\n'...
    '                int cvodewrap_SetJacTimesVecFn(void *cvode_mem){\n'...
    '                    return CVSpilsSetJacTimesVecFn(cvode_mem, Jv_' filename ');\n'...
    '                }\n'...
    '                int cvodewrap_SetDenseJacFnB(void *cvode_mem,int which){\n'...
    '                    return CVDlsSetDenseJacFnB(cvode_mem, which, JB_' filename ');\n'...
    '                }\n'...
    '                int cvodewrap_SetSparseJacFnB(void *cvode_mem, int which){\n'...
    '                    return CVSlsSetSparseJacFnB(cvode_mem, which, JSparseB_' filename ');\n'...
    '                }\n'...
    '                int cvodewrap_SetBandJacFnB(void *cvode_mem,int which){\n'...
    '                    return CVDlsSetBandJacFnB(cvode_mem, which, JBBand_' filename ');\n'...
    '                }\n'...
    '                int cvodewrap_SetJacTimesVecFnB(void *cvode_mem,int which){\n'...
    '                    return CVSpilsSetJacTimesVecFnB(cvode_mem, which, JvB_' filename ');\n'...
    '                }\n'...
    '                int fJ(long int N, realtype t, N_Vector x,N_Vector fx, DlsMat J, void *user_data,N_Vector tmp1, N_Vector tmp2, N_Vector tmp3){\n'...
    '                    return J_' filename '(N,t,x,fx,J,user_data,tmp1,tmp2,tmp3);\n'...
    '                }\n'...
    '                int fJB(long int N, realtype t, N_Vector x, N_Vector xB, N_Vector fx, DlsMat J, void *user_data,N_Vector tmp1, N_Vector tmp2, N_Vector tmp3){\n'...
    '                    return JB_' filename '(N,t,x,xB,fx,J,user_data,tmp1,tmp2,tmp3);\n'...
    '                }\n'...
    '                \n'...
    '                void fsx0(N_Vector *sx0, void *user_data){\n'...
    '                    UserData data = (UserData) user_data;\n'...
    '                    int ip;\n'...
    '                    int *plist = data->plist;\n'...
    '                    int np = *data->np;\n'...
    '                    for (ip=0; ip<np; ip++) {\n'...
    '                       sx0_' filename '(plist[ip], sx0[ip], data);\n'...
    '                    }\n'...
    '                }\n'...
    '                \n'...
    '                int cvodewrap_SensInit1(void *cvode_mem, int np, int sensi_meth, N_Vector *sx){\n'...
    '                    return CVodeSensInit1(cvode_mem, np, sensi_meth, sx_' filename ', sx);\n'...
    '                }\n'...
    '                int cvodewrap_RootInit(void *cvode_mem, int nr){\n'...
    '                    return CVodeRootInit(cvode_mem, nr, root_' filename ');\n'...
    '                }\n'...
    '                void froot(double t, N_Vector x, realtype *gout, void *user_data){\n'...
    '                    root_' filename '(t, x, gout, user_data);\n'...
    '                }\n'...
    '                \n'...
    '                void fy(double t, int nt, int it, double *y, double *p, double *k, double *u, double *x){\n'...
    '                    y_' filename '(t, nt, it, y, p, k, u, x);\n'...
    '                }\n'...
    '                \n'...
    '                void fxdot(realtype t, N_Vector x, N_Vector xdot, void *user_data){\n'...
    '                    xdot_' filename '(t,x,xdot,user_data);\n'...
    '                }\n'...
    '                void fdydp(double t, int nt, int it,double *dydp, double *y, double *p, double *k, double *u, double *x, int *plist, int np, int ny){\n'...
    '                    dydp_' filename '(t, nt, it, dydp, y, p, k, u, x, plist, np, ny);\n'...
    '                }\n'...
    '                void fdxdotdp(double t, int nt, int it,double *dxdotdp, double *p, double *k, double *u, double *x, int *plist, int np, int nx){\n'...
    '                    dxdotdp_' filename '(t, nt, it, dxdotdp, p, k, u, x, plist, np, nx);\n'...
    '                }\n'...
    '                void fdydx(double t,double *dydx, double *y, double *p, double *k, double *x){\n'...
    '                    dydx_' filename '(t, dydx, y, p, k, x);\n'...
    '                }\n'...
    '                \n'...
    '                \n'...
    '                void fsy(double t, int nt, int it, int *plist, int nx, int ny, int np, double *sy, double *p, double *k, double *x, double *sx){\n'...
    '                    int ip;\n'...
    '                    for (ip=0; ip<np; ip++) {\n'...
    '                        sy_' filename '(t, nt, it, plist[ip], ip,  nx, ny,  sy, p, k, x, sx);\n'...
    '                    }\n'...
    '                }\n'...
    '                double fsroot(double t, int ip, int ir, N_Vector x, N_Vector sx, void *user_data){\n'...
    '                    return(sroot_' filename '(t, ip, ir, x, sx, user_data));\n'...
    '                }\n'...
    '                double fsrootval(double t, int ip, int ir, N_Vector x, N_Vector sx, void *user_data){\n'...
    '                    return(srootval_' filename '(t, ip, ir, x, sx, user_data));\n'...
    '                }\n'...
    '                double fs2root(double t, int ip, int jp, int ir, N_Vector x, N_Vector sx, void *user_data){\n'...
    '                    return(s2root_' filename '(t, ip, jp, ir, x, sx, user_data));\n'...
    '                }\n'...
    '                double fs2rootval(double t, int ip, int jp, int ir, N_Vector x, N_Vector sx, void *user_data){\n'...
    '                    return(s2rootval_' filename '(t, ip, jp, ir, x, sx, user_data));\n'...
    '                }\n'...
    '                void deltadisc(double t, int idisc, N_Vector x, void *user_data){\n'...
    '                       deltadisc_' filename '(t, idisc, x, user_data);\n'...
    '                }\n'...
    '                void sdeltadisc(double t, int idisc, N_Vector x, N_Vector *sx, void *user_data){\n'...
    '                    UserData data = (UserData) user_data;\n'...
    '                    sdeltadisc_' filename '(t, idisc, x, sx, data);\n'...
    '                }\n'...
    '                void cvodewrap_ErrHandlerFn(int error_code, const char *module, const char *function, char *msg, void *eh_data){\n'...
    '                    char buffer [250];\n'...
    '                    sprintf(buffer,"CVODES ERROR: in module %%s in function %%s : %%s ",module,function,msg);\n'...
    '                    mexWarnMsgTxt(buffer);\n'...
    '                }\n\n'...
    ];

% write the string to the file
fid = fopen(fullfile(odewrap_path,'models',filename,'cvodewrapfunctions.c'),'w');
fprintf(fid,str_funcfile);
fclose(fid);

fprintf('\r');

end


function writeCcode_sensi(struct,svar,fid)

np = length(struct.sym.p);

if(strcmp(svar,'sx'))
    nonzero = struct.sym.sx~=0;
    if(any(any(nonzero)))
        for ip=1:np
            if(any(nonzero(:,ip)))
                fprintf(fid,['  case ' num2str(ip-1) ': {\n']);
                writeCcode(struct,'sx', fid, ip)
                fprintf(fid,'\n');
                fprintf(fid,'  } break;\n\n');
            end
        end
    end
elseif(strcmp(svar,'sx0'))
    nonzero = struct.sym.sx0~=0;
    if(any(any(nonzero)))
        for ip=1:np
            if(any(nonzero(:,ip)))
                fprintf(fid,['  case ' num2str(ip-1) ': {\n']);
                writeCcode(struct,'sx0', fid, ip)
                fprintf(fid,'\n');
                fprintf(fid,'  } break;\n\n');
            end
        end
    end
elseif(strcmp(svar,'sy'))
    nonzero = struct.sym.sy~=0;
    if(any(any(nonzero)))
        for ip=1:np
            if(any(nonzero(:,ip)))
                fprintf(fid,['  case ' num2str(ip-1) ': {\n']);
                writeCcode(struct,'sy', fid, ip)
                fprintf(fid,'\n');
                fprintf(fid,'  } break;\n\n');
            end
        end
    end
elseif(strcmp(svar,'xQB'))
    nonzero = struct.sym.int~=0;
    if(any(any(nonzero)))
        for ip=1:np
            if(any(nonzero(:,ip)))
                fprintf(fid,['  case ' num2str(ip-1) ': {\n']);
                writeCcode(struct,'xQB', fid, ip)
                fprintf(fid,'\n');
                fprintf(fid,'  } break;\n\n');
            end
        end
    end
elseif(strcmp(svar,'dydp'))
    nonzero = struct.sym.dydp~=0;
    if(any(any(struct.sym.dydp~=0)))
        for ip=1:np
            if(any(nonzero(:,ip)))
                fprintf(fid,['  case ' num2str(ip-1) ': {\n']);
                writeCcode(struct,'dydp', fid, ip)
                fprintf(fid,'\n');
                fprintf(fid,'  } break;\n\n');
            end
        end
    end
elseif(strcmp(svar,'sroot'))
    nonzero = struct.sym.sroot~=0;
    if(any(any(nonzero)))
        for ip=1:np
            if(any(nonzero(:,ip)))
                fprintf(fid,['  case ' num2str(ip-1) ': {\n']);
                writeCcode(struct,'sroot', fid, ip)
                fprintf(fid,'\n');
                fprintf(fid,'  } break;\n\n');
            end
        end
    end
elseif(strcmp(svar,'srootval'))
    nonzero = struct.sym.sroot~=0;
    if(any(any(nonzero)))
        for ip=1:np
            if(any(nonzero(:,ip)))
                fprintf(fid,['  case ' num2str(ip-1) ': {\n']);
                writeCcode(struct,'srootval', fid, ip)
                fprintf(fid,'\n');
                fprintf(fid,'  } break;\n\n');
            end
        end
    end
elseif(strcmp(svar,'s2root'))
    nonzero = struct.sym.s2root~=0;
    if(any(any(any(nonzero))))
        for ip=1:np
            if(any(any(nonzero(:,:,ip))))
                fprintf(fid,['  case ' num2str(ip-1) ': {\n']);
                fprintf(fid,['      switch(jp) {\n']);
                for jp=1:np
                    if(any(nonzero(:,jp,ip)))
                        fprintf(fid,['          case ' num2str(jp-1) ': {\n']);
                        writeCcode(struct,'s2root', fid, ip, jp)
                        fprintf(fid,'\n');
                        fprintf(fid,'          } break;\n\n');
                    end
                end
                fprintf(fid,'\n      }\n');
                fprintf(fid,'  } break;\n\n');
            end
        end
    end
elseif(strcmp(svar,'s2rootval'))
    nonzero = struct.sym.s2rootval~=0;
    if(any(any(any(nonzero))))
        for ip=1:np
            if(any(any(nonzero(:,:,ip))))
                fprintf(fid,['  case ' num2str(ip-1) ': {\n']);
                fprintf(fid,['      switch(jp) {\n']);
                for jp=1:np
                    if(any(nonzero(:,jp,ip)))
                        fprintf(fid,['          case ' num2str(jp-1) ': {\n']);
                        writeCcode(struct,'s2rootval', fid, ip, jp)
                        fprintf(fid,'\n');
                        fprintf(fid,'           } break;\n\n');
                    end
                end
                fprintf(fid,'\n      }\n');
                fprintf(fid,'  } break;\n\n');
            end
        end
    end
elseif(strcmp(svar,'sdeltadisc'))
    nonzero = struct.sym.sdeltadisc~=0;
    if(any(any(any(nonzero))))
        for ip=1:np
            if(any(any(nonzero(:,ip,:))))
                fprintf(fid,['  case ' num2str(ip-1) ': {\n']);
                writeCcode(struct,'sdeltadisc', fid, ip)
                fprintf(fid,'  } break;\n\n');
            end
        end
    end
elseif(strcmp(svar,'dxdotdp'))
    nonzero = struct.sym.dxdotdp~=0;
    if(any(any(struct.sym.dxdotdp~=0)))
        for ip=1:np
            if(any(nonzero(:,ip)))
                fprintf(fid,['  case ' num2str(ip-1) ': {\n']);
                writeCcode(struct,'dxdotdp', fid, ip)
                fprintf(fid,'\n');
                fprintf(fid,'  } break;\n\n');
            end
        end
    end
end
end


function writeCcode(struct, svar, fid, ip, jp)

nx = struct.nx;
ny = struct.ny;
nr = struct.nr;
ndisc = struct.ndisc;
if(strcmp(svar,'xdot'))
    fprintf(fid,['  memset(xdot_tmp,0,sizeof(double)*' num2str(nx) ');\n']);
    cvar =  'xdot_tmp';
    gccode(struct.sym.xdot(:),cvar,fid);
elseif(strcmp(svar,'roots'))
    if(nr+ndisc>0)
        fprintf(fid,['  memset(gout,0,sizeof(double)*' num2str(nr+ndisc) ');\n']);
        cvar =  'gout';
        gccode(struct.sym.root(:),cvar,fid);
    end
elseif(strcmp(svar,'x0'))
    fprintf(fid,['  memset(x0_tmp,0,sizeof(double)*' num2str(nx) ');\n']);
    cvar =  'x0_tmp';
    gccode(struct.sym.x0(:),cvar,fid);
elseif(strcmp(svar,'J'))
    fprintf(fid,['  memset(J->data,0,sizeof(double)*' num2str(nx^2) ');\n']);
    cvar =  'Jdata';
    gccode(struct.sym.J(:),cvar,fid);
elseif(strcmp(svar,'JSparse'))
    cvar =  'Jdata';
    gccode(struct.sym.J(struct.sparseidx),cvar,fid);
elseif(strcmp(svar,'JB'))
    cvar =  'JBdata';
    gccode(struct.sym.JB(:),cvar,fid);
elseif(strcmp(svar,'JBsparse'))
    cvar =  'JBdata';
    gccode(struct.sym.JB(struct.sparseidxB),cvar,fid);
elseif(strcmp(svar,'Jv'))
    fprintf(fid,['  memset(Jv_tmp,0,sizeof(double)*' num2str(nx) ');\n']);
    cvar =  'Jv';
    gccode(struct.sym.Jv(:),cvar,fid);
elseif(strcmp(svar,'JvB'))
    fprintf(fid,['  memset(JvB_tmp,0,sizeof(double)*' num2str(nx) ');\n']);
    cvar =  'JvB';
    gccode(struct.sym.JvB(:),cvar,fid);
elseif(strcmp(svar,'sx'))
    cvar =  'sxdot_tmp';
    if(any(struct.sym.sx(:,ip)~=0))
        gccode(struct.sym.sx(:,ip),cvar,fid);
    end
elseif(strcmp(svar,'sx0'))
    cvar =  'sx0_tmp';
    if(any(struct.sym.sx0(:,ip)~=0))
        gccode(struct.sym.sx0(:,ip),cvar,fid);
    end
elseif(strcmp(svar,'y'))
    cvar =  'y';
    gccode(struct.sym.y(:),cvar,fid);
elseif(strcmp(svar,'dydp'))
    cvar =  'dydp';
    if(any(struct.sym.dydp(:,ip)~=0))
        gccode(struct.sym.dydp(:,ip),cvar,fid);
    end
elseif(strcmp(svar,'dydx'))
    fprintf(fid,['  memset(dydx,0,sizeof(double)*' num2str(ny*nx) ');\n']);
    cvar =  'dydx';
    gccode(struct.sym.dydx(:),cvar,fid);
elseif(strcmp(svar,'xB0'))
    fprintf(fid,['  memset(xB0,0,sizeof(double)*' num2str(ny*nx) ');\n']);
    cvar =  'xB0';
    gccode(struct.sym.xB0(:),cvar,fid);
elseif(strcmp(svar,'sy'))
    cvar =  'sy';
    if(any(struct.sym.sy(:,ip)~=0))
        gccode(struct.sym.sy(:,ip),cvar,fid);
    end
elseif(strcmp(svar,'sroot'))
    cvar =  'dr_dp';
    nonzero = struct.sym.sroot(:,ip)~=0;
    if(any(nonzero))
        fprintf(fid,'          switch(ir) {\n');
        for ir=1:nr
            if(nonzero(ir))
                fprintf(fid,['          case ' num2str(ir-1) ': {\n']);
                gccode(struct.sym.sroot(ir,ip),cvar,fid);
                fprintf(fid,'\n');
                fprintf(fid,'          } break;\n\n');
            end
        end
        fprintf(fid,'          }\n');
    end
    
elseif(strcmp(svar,'s2root'))
    cvar =  'ddr_dpdp';
    nonzero = struct.sym.s2root(:,ip,jp)~=0;
    if(any(nonzero))
        fprintf(fid,'              switch(ir)\n');
        for ir=1:nr
            if(nonzero(ir))
                fprintf(fid,['              case ' num2str(ir-1) ': {\n']);
                gccode(struct.sym.s2root(ir,ip,jp),cvar,fid);
                fprintf(fid,'\n');
                fprintf(fid,'              } break;\n\n');
            end
        end
    end
elseif(strcmp(svar,'srootval'))
    cvar =  'dg_dp';
    nonzero = struct.sym.srootval(:,ip)~=0;
    if(any(nonzero))
        fprintf(fid,'          switch(ir) { \n');
        for ir=1:nr
            if(nonzero(ir))
                fprintf(fid,['          case ' num2str(ir-1) ': {\n']);
                gccode(struct.sym.srootval(ir,ip),cvar,fid);
                fprintf(fid,'\n');
                fprintf(fid,'          } break;\n\n');
            end
        end
        fprintf(fid,'          } \n');
    end
elseif(strcmp(svar,'s2rootval'))
    cvar =  'ddg_dpdp';
    nonzero = struct.sym.s2rootval(:,ip,jp) ~=0;
    if(any(nonzero))
        fprintf(fid,'              switch(ir) { \n');
        for ir=1:nr
            if(nonzero(ir)~=0)
                fprintf(fid,['              case ' num2str(ir-1) ': {\n']);
                gccode(struct.sym.s2rootval(ir,ip,jp),cvar,fid);
                fprintf(fid,'\n');
                fprintf(fid,'              } break;\n\n');
            end
        end
        fprintf(fid,'              } \n');
    end
elseif(strcmp(svar,'deltadisc'))
    cvar =  'deltadisc';
    nonzero = struct.sym.deltadisc ~=0;
    if(any(any(nonzero)))
        fprintf(fid,'              switch(idisc) { \n');
        for idisc=1:ndisc
            if(any(nonzero(:,idisc)~=0))
                fprintf(fid,['              case ' num2str(idisc-1) ': {\n']);
                gccode(struct.sym.deltadisc(:,idisc),cvar,fid);
                fprintf(fid,'\n');
                fprintf(fid,'              } break;\n\n');
            end
        end
        fprintf(fid,'              } \n');
    end
elseif(strcmp(svar,'sdeltadisc'))
    cvar =  'sdeltadisc';
    nonzero = struct.sym.sdeltadisc(:,ip,:) ~=0;
    if(any(any(nonzero)))
        fprintf(fid,'              switch(idisc) { \n');
        for idisc=1:ndisc
            if(any(nonzero(:,idisc)~=0))
                fprintf(fid,['              case ' num2str(idisc-1) ': {\n']);
                gccode(struct.sym.sdeltadisc(:,ip,idisc),cvar,fid);
                fprintf(fid,'\n');
                fprintf(fid,'              } break;\n\n');
            end
        end
        fprintf(fid,'              } \n');
    end
elseif(strcmp(svar,'xB'))
    fprintf(fid,['  memset(xBdot_tmp,0,sizeof(double)*' num2str(ny*nx) ');\n']);
    cvar =  'xBdot_tmp';
    gccode(struct.sym.xBdot(:),cvar,fid);
    
elseif(strcmp(svar,'xQB'))
    cvar =  'qBdot_tmp';
    gccode(struct.sym.int(:,ip),cvar,fid);
elseif(strcmp(svar,'dxdotdp'))
    cvar =  'dxdotdp';
    if(any(struct.sym.dxdotdp(:,ip)~=0))
        gccode(struct.sym.dxdotdp(:,ip),cvar,fid);
    end
    
end

end

function gccode(csym,cvar,fid)

lcsym = length(csym);
for j=transpose(find(csym~=0));
    % temp = sprintf(mupadmex('generate::C', [cvar '[' num2str(j) '] =
    % ' char(csym(j))], 0)); this is much too slow for large systems
    if( lcsym == 1)
        cstr = [cvar ' = ' char(vpa(csym)) ';'];
%         tmp_str = ccode(csym(j));
%         cstr = [ cvar tmp_str(5:end) ';'];
    else
        cstr = [cvar '[' num2str(j-1) '] = ' char(vpa(csym(j))) ';'];
%         tmp_str = ccode(csym(j));
%         cstr = [ cvar '[' num2str(j-1) '] ' tmp_str(5:end) ';'];
    end

    
    % parse pow(...,...)
    idx = strfind(cstr,'^');
    if(~isempty(idx))
        bro = regexp(cstr,'\(');%find bracket opening
        brc = regexp(cstr,'\)');%find bracket closing
        % compute nested bracket level of expression
        brl = zeros(length(cstr),1);
        l = 0;
        for k = 1:length(cstr)
            if(~isempty(find(k==bro,1)))
                l = l+1;
            end
            brl(k) = l;
            if(~isempty(find(k==brc,1)))
                l = l-1;
            end
        end
    end
    while(~isempty(idx))
        samebrl = find(brl == brl(idx(1))); % find bracket level same as found level
        % find first argument
        if(strcmp(cstr(idx(1)-1),')'))% if the expression is braced find start of higher levels
            arg1start = samebrl(find(samebrl<idx(1),1,'last')) + 1;
            while(~isempty(regexp(cstr(arg1start-1),'[\w]','once')))
                arg1start = arg1start-1;
            end
        else % otherwise find start of expression
            m = regexp(cstr(1:(idx(1)-1)),'([\w\[\]]*)','start');
            arg1start = m(end);
        end
        arg1end = idx(1)-1;
        % find second argument
        arg2start = idx(1)+1;
        if(strcmp(cstr(idx(1)+1),'('))% if the expression is braced find end of higher levels
            arg2end = samebrl(find(samebrl>idx(1),1,'first')) - 1;
        else % otherwise find end of expression
            m = regexp(cstr((idx(1)+1):end),'([\w\[\]]*)','end');
            arg2end = idx(1)+m(1);
        end
        
        % recombine
        cstr = [cstr(1:(arg1start-1)) '(pow('                                                    cstr(arg1start:arg1end)   ','              cstr(arg2start:arg2end)                 '))'             cstr((arg2end+1):end)];
        % compute new bracket level
        brl  = [ brl(1:(arg1start-1)); brl(idx(1))+1; ones(3,1)*brl(idx(1))+1 ; brl(idx(1))+2 ; brl(arg1start:arg1end)+2 ; brl(idx(1))+2 ; brl(arg2start:arg2end)+2 ; brl(idx(1))+2; brl(idx(1))+1; brl((arg2end+1):end) ];
        idx = strfind(cstr,'^');
    end
    cstr = deblank(cstr);
    cstr = regexprep(cstr,'D\(\[([0-9]*)\]\, ([\w]*)\)\(','D$2\($1,'); % fix derivatives
    cstr = regexprep(cstr,'dirac\(([1-2]*),','Ddirac\($1,'); % fix derivatives
    cstr = regexprep(cstr,'dirac\(([^,]*), ([1-2]*)\)','Ddirac\($2,$1\)'); % fix matlab <= R2014a
    cstr = regexprep(cstr,'abs\(','fabs\('); % fix abs->fabs
    
    
    if(strcmp(cvar,'qBdot_tmp'))
        cstr = regexprep(cstr,'qBdot_tmp\[([0-9]*)\]','qBdot_tmp\[$1+ip*ny]');
        cstr = strrep(cstr,'qBdot_tmp ','qBdot_tmp[ip] ');
    end
    
    if(~(length(cstr)==1 && isempty(cstr{1})))
        if(strcmp(cvar,'y') || strcmp(cvar,'sy') || strcmp(cvar,'dydp') || strcmp(cvar,'dxdotdp'))
            if(strcmp(cvar,'y') || strcmp(cvar,'dydp') || strcmp(cvar,'dxdotdp'))
                cstr = strrep(cstr, 'dydp ', 'dydp[0] ');
                cstr = strrep(cstr, 'dxdotdp ', 'dxdotdp[0] ');
                cstr = strrep(cstr, 'y ', 'y[0] ');
                cstr = strrep(cstr, 'x[', 'x[it+nt*');
                cstr = strrep(cstr, 'y[', 'y[it+nt*');
                cstr = regexprep(cstr, 'dydp\[([0-9]*)\]', 'dydp[it+nt*\($1+ip*ny\)\]');
                cstr = regexprep(cstr, 'dxdotdp\[([0-9]*)\]', 'dxdotdp[it+nt*\($1+ip*nx\)\]');
            else
                cstr = strrep(cstr,'sy ','sy[0] ');
                cstr = regexprep(cstr,'sx\[([0-9]*)\]','sx\[it+nt*\($1+np*nx\)\]');
                cstr = regexprep(cstr,'sy\[([0-9]*)\]','sy\[it+nt*\($1+np*ny\)\]');
                cstr = regexprep(cstr,'y\[([0-9]*)\]','y\[it+nt*$1\]');
                cstr = regexprep(cstr,'x\[([0-9]*)\]','x\[it+nt*$1\]');
            end
            cstr = strrep(cstr, 'u[', 'u[it+nt*');
        elseif(strcmp(cvar,'sdeltadisc'))
            cstr = regexprep(cstr,'sdeltadisc\[([0-9]*)\]','sdeltadisc\[plist[ip]+np*$1\]');
            cstr = strrep(cstr, 'x[', 'x_tmp[');
        elseif(strcmp(cvar,'dydx'))
            % do nothing
        elseif(strcmp(cvar,'Jdata'))
            cstr = strrep(cstr, 'Jdata[', 'J->data[');
            cstr = strrep(cstr, 'x[', 'x_tmp[');
        elseif(strcmp(cvar,'JBdata'))
            cstr = strrep(cstr, 'JBdata[', 'JB->data[');
            cstr = strrep(cstr, 'x[', 'x_tmp[');
            cstr = strrep(cstr, 'xB[', 'xB_tmp[');
        elseif(strcmp(cvar,'Jv'))
            cstr = strrep(cstr, 'v[', 'v_tmp[');
            cstr = strrep(cstr, 'x[', 'x_tmp[');
        elseif(strcmp(cvar,'JvB'))
            cstr = strrep(cstr, 'v[', 'vB_tmp[');
            cstr = strrep(cstr, 'JvB[', 'JvB_tmp[');
            cstr = strrep(cstr, 'x[', 'x_tmp[');
            cstr = strrep(cstr, 'xB[', 'xB_tmp[');
        elseif(strcmp(cvar,'gout'))
            cstr = strrep(cstr, 'gout =', 'gout[0] =');
            cstr = strrep(cstr, 'x[', 'x_tmp[');
        elseif(strcmp(cvar,'sx0_tmp'))
            cstr = strrep(cstr, 'sx0_tmp =', 'sx0_tmp[0] =');
        else
            cstr = strrep(cstr, 'x[', 'x_tmp[');
            cstr = strrep(cstr, 'xB[', 'xB_tmp[');
        end
    end
    fprintf(fid,[cstr '\n']);
end
end