%function [logL,grad] = logLikelihoodSSE(xi,D,model,options)
function varargout = logLikelihoodSSE(varargin)
    
    %% CHECK/ASSIGN INPUTS:
    if nargin >= 3
        xi = varargin{1};
        D = varargin{2};
        model = varargin{3};
    else
        error('Not enough inputs!');
    end
    
    % Assign defaults for options
    options.sign = 'positive';
    options.grad_ind = [1:length(xi)]';
    if nargin == 4
        options = setdefault(varargin{4},options);
    end
    
    try
        %% Parameter transformation
        theta = 10.^xi;

        %% Simulation
        options_simu.sensi = 1;
        options_simu.cvodes_maxsteps = 100000;
        switch options.model
            case 'model_1'
                switch model
                    case 'IOS'
                        sol = simulate_model_1_IOS(D.t,theta,[],options_simu);
                    case 'EMRE'
                        sol = simulate_model_1_EMRE(D.t,theta,[],options_simu);
                    case 'LNA'
                        sol = simulate_model_1_LNA(D.t,theta,[],options_simu);
                    case 'RRE'
                        sol = simulate_model_1_RRE(D.t,theta,[],options_simu);
                    case '2MA'
                        sol = simulate_model_1_2MA(D.t,theta,[],options_simu);
                    case '3MA'
                        sol = simulate_model_1_3MA(D.t,theta,[],options_simu);
                end
            case 'model_trimerization'
                switch model
                    case 'IOS'
                        sol = simulate_model_trimerization_IOS(D.t,theta,[],options_simu);
                    case 'EMRE'
                        sol = simulate_model_trimerization_EMRE(D.t,theta,[],options_simu);
                    case 'LNA'
                        sol = simulate_model_trimerization_LNA(D.t,theta,[],options_simu);
                    case 'RRE'
                        sol = simulate_model_trimerization_RRE(D.t,theta,[],options_simu);
                    case '2MA'
                        sol = simulate_model_trimerization_2MA(D.t,theta,[],options_simu);
                    case '3MA'
                        sol = simulate_model_trimerization_3MA(D.t,theta,[],options_simu);
                end
        end
        
        y = sol.y;
        sy = sol.sy;
        status = sol.status;

        y_ind = options.y_ind;

        my = y(:,y_ind);
        
        
        dmydtheta = sy(:,y_ind,:);
        dmydxi = bsxfun(@times,dmydtheta,permute(theta,[3,2,1])*log(10));

        for j = 1:3
            D.sigma_my(D.sigma_my(:,j)==0,j) = min(D.sigma_my(D.sigma_my(:,j)>0,j))/1e2;
        end

        if(any([strcmp(model,'LNA'),strcmp(model,'IOS'),strcmp(model,'3MA')]))
            Cy = y(:,3+y_ind);
            dCydtheta = sy(:,3+y_ind,:);
            dCydxi = bsxfun(@times,dCydtheta,permute(theta,[3,2,1])*log(10));
            for j = 1:3
                D.sigma_Cy(D.sigma_Cy(:,j)==0,j) = min(D.sigma_Cy(D.sigma_Cy(:,j)>0,j))/1e2;
            end
        end


        %% Log-likelihood function
        logL = 0;
        grad = zeros(length(xi),1);
        fish = zeros(length(xi),length(xi));

        % Mean
        nt = size(my,1);
        nm = size(my,2);
        np = size(dmydxi,3);
        resmy = reshape((my(:,:)-D.my(:,y_ind))./(D.sigma_my(:,y_ind)),nt*nm,1);
        sresmy = reshape(bsxfun(@times,dmydxi(:,:,:),1./(D.sigma_my(:,y_ind))),nt*nm,np);
        logL = logL - 0.5*sum(resmy.^2);
        grad = grad - sresmy'*resmy;
        fish = fish - sresmy'*sresmy;
        
        if(any([strcmp(model,'LNA'),strcmp(model,'IOS'),strcmp(model,'3MA')]))
            nC = size(Cy,2);
            resCy = reshape((Cy(:,:)-D.Cy(:,y_ind))./(D.sigma_Cy(:,y_ind)),nt*nC,1);
            sresCy = reshape(bsxfun(@times,dCydxi(:,:,:),1./(D.sigma_Cy(:,y_ind))),nt*nm,np);
            logL = logL - 0.5*sum(resCy.^2);
            grad = grad - sresCy'*resCy;
            fish = fish - sresCy'*sresCy;
        end
        %Error catch
    catch
        logL = -inf;
        grad = zeros(length(xi),1);
        fish = zeros(length(xi),length(xi));
    end

    if status < 0
        error('failed to integrate')
    end
    
    %% ASSIGN OUTPUT
    switch nargout
        % One output
        case {0,1}
            switch  options.sign
                case 'positive'
                    varargout{1} =  logL;
                case 'negative'
                    varargout{1} = -logL;
            end
            % Two outputs
        case 2
            switch  options.sign
                case 'positive'
                    varargout{1} =  logL;
                    varargout{2} =  grad(options.grad_ind);
                case 'negative'
                    varargout{1} = -logL;
                    varargout{2} = -grad(options.grad_ind);
            end
        case 3
            switch  options.sign
                case 'positive'
                    varargout{1} =  logL;
                    varargout{2} =  grad(options.grad_ind);
                    varargout{3} =  fish(options.grad_ind,options.grad_ind);
                case 'negative'
                    varargout{1} = -logL;
                    varargout{2} = -grad(options.grad_ind);
                    varargout{3} = -fish(options.grad_ind,options.grad_ind);
            end
            
    end